/*
 * @(#)OMObjectFigure.java  2.0  2006-01-16
 *
 * Copyright (c) 2003-2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.oo.diagram;

import ch.hslu.cm.*;
import ch.hslu.cm.oo.*;
import ch.hslu.cm.oo.objectmodel.*;
import java.awt.geom.Point2D.Double;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;
import javax.swing.*;
import org.jhotdraw.app.action.ActionUtil;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.connector.ChopRectangleConnector;
import org.jhotdraw.draw.connector.Connector;
import org.jhotdraw.draw.connector.StickyRectangleConnector;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.HorizontalLayouter;
import org.jhotdraw.draw.layouter.VerticalLayouter;
import org.jhotdraw.geom.*;
import org.jhotdraw.util.ResourceBundleUtil;

/**
 * OMObjectFigure visually represents a {@link OMObject}.
 * 
 * @author Werner Randelshofer
 * @version 2.0 2006-01-16 Changed to support double coordinates.
 * <br>1.0 1. Dezember 2003  Created.
 */
public class OMObjectFigure
        extends GraphicalCompositeFigure
        implements DiagramFigure, OMObjectListener, OMClassListener, PropertyChangeListener {

    private OMObject model;
    /*private ListFigure nameCompartment;
    private ListFigure attributeCompartment;*/
    private OMClass clazz;

    /**
     * This adapter is used, to connect a TextFigure with the name of
     * the OMClass model.
     */
    private static class NameAdapter extends FigureAdapter {

        private OMObjectFigure target;

        public NameAdapter(OMObjectFigure target) {
            this.target = target;
        }

        @Override
        public void attributeChanged(FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                target.model.setName((String) e.getNewValue());
            }
        }
    }

    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the SimulatedObject model.
     */
    private class VariableAdapter extends FigureAdapter {

        private OMAttribute attr;

        public VariableAdapter(OMAttribute attr) {
            this.attr = attr;
        }

        @Override
        public void attributeChanged(FigureEvent evt) {
            if (evt.getAttribute().equals(TEXT)) {
                model.set(attr, getSimulation().unwrapValue((String) evt.getNewValue()));
            }
        }
    }

    @Override
    public Connector findCompatibleConnector(Connector c, boolean isStart) {
        if (c instanceof StickyRectangleConnector) {
            Point2D.Double p = c.getAnchor();
            return new StickyRectangleConnector(this, p);
        } else {
            return new ChopRectangleConnector(this);
        }
    }

    /**
     * Gets a connector for this figure at the given location.
     * A figure can have different connectors at different locations.
     */
    @Override
    public Connector findConnector(Point2D.Double p, ConnectionFigure prototype) {
        return new StickyRectangleConnector(this, p);
    }

    /** Creates a new instance. */
    public OMObjectFigure() {
        //super(new ShadowRectangleFigure());
        super(new RectangleFigure());

        setLayouter(new VerticalLayouter());
        ListFigure nameCompartment = new ListFigure();
        nameCompartment.setLayouter(new HorizontalLayouter());
        ListFigure attributeCompartment = new ListFigure();

        add(nameCompartment);
        add(new SeparatorLineFigure());
        add(attributeCompartment);

        Insets2D.Double insets = new Insets2D.Double(4, 6, 4, 6);
        LAYOUT_INSETS.set(nameCompartment, insets);
        LAYOUT_INSETS.set(attributeCompartment, insets);

        TextFigure nameFigure;
        nameCompartment.add(nameFigure = new TextFigure());
        nameFigure.addFigureListener(new NameAdapter(this));
        FONT_UNDERLINE.set(nameFigure, true);
        nameCompartment.add(nameFigure = new TextFigure());
        nameFigure.setEditable(false);
        FONT_UNDERLINE.set(nameFigure, true);

        setModel(createSimulatedObject());

        setAttributeEnabled(STROKE_DASHES, false);
    }

    protected OMObject createSimulatedObject() {
        return new OMObject();
    }

    public void setModel(OMObject m) {
        willChange();
        if (model != null) {
            model.removeSimulatedObjectListener(this);
            model.removePropertyChangeListener(this);
            if (model.getSimulatedClass() != null) {
                model.getSimulatedClass().removeSimulatedClassListener(this);
            }
        }
        if (clazz != null) {
            clazz.removeSimulatedClassListener(this);
        }
        model = m;
        clazz = (m == null) ? null : m.getSimulatedClass();

        if (model != null) {
            model.addSimulatedObjectListener(this);
            model.addPropertyChangeListener(this);
            if (model.getSimulatedClass() != null) {
                model.getSimulatedClass().addSimulatedClassListener(this);
            }
        }
        if (clazz != null) {
            clazz.addSimulatedClassListener(this);
        }
        updateNameCompartment();
        updateAttributeCompartment();
        layout();
        changed();
    }

    @Override
    public OMObject getModel() {
        return model;
    }

    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;

        //Figure compartment = findCompartment(x, y);
        Figure item;

        Collection classes = model.getObjectModel().getSimulatedClasses();
        for (Iterator i = classes.iterator(); i.hasNext();) {
            final OMClass clazz = (OMClass) i.next();
            action = new AbstractAction(clazz.getName()) {

                @Override
                public void actionPerformed(ActionEvent event) {
                    System.out.println("ObjectFigure setting clazz:" + clazz);
                    model.setSimulatedClass(clazz);
                }
            };
            ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");
            action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.class.text"));
            actions.add(action);
        }
        return actions;
    }

    private ListFigure getNameCompartment() {
        return (ListFigure) getChild(0);
    }

    private ListFigure getAttributeCompartment() {
        return (ListFigure) getChild(2);
    }

    private void updateNameCompartment() {
        TextFigure nameFigure = (TextFigure) getNameCompartment().getChild(0);
        TextFigure classFigure = (TextFigure) getNameCompartment().getChild(1);
        OMClass clazz = model.getSimulatedClass();
        nameFigure.setText(model.getName());
        classFigure.setText((clazz == null) ? "" : ":" + clazz.getName());

    }

    private void updateAttributeCompartment() {
        getAttributeCompartment().removeAllChildren();
        OMClass clazz = model.getSimulatedClass();
        if (clazz != null) {
            for (Iterator i = clazz.getAllAttributes().iterator(); i.hasNext();) {
                OMAttribute attr = (OMAttribute) i.next();
                ListFigure listf = new ListFigure();
                listf.set(CompositeFigure.LAYOUT_INSETS, new Insets2D.Double(1, 0, 2, 0));
                LAYOUT_INSETS.set(listf, new Insets2D.Double(0, 0, 0, 0));
                listf.setLayouter(new HorizontalLayouter());
                LabelFigure labelf = new LabelFigure(attr.getAttributeName() + "=");
                listf.add(labelf);
                String value = (getSimulation() == null) ? "" + model.get(attr) : getSimulation().wrapValue(model.get(attr));
                TextFigure txtf = new TextFigure(value);
                labelf.setLabelFor(txtf);
                listf.add(txtf);
                txtf.addFigureListener(new VariableAdapter(attr));
                getAttributeCompartment().add(listf);
            }
        }
    }

    public Figure findCompartment(Point2D.Double p) {
        if (getNameCompartment().contains(p)) {
            return getNameCompartment();
        }
        if (getAttributeCompartment().contains(p)) {
            return getAttributeCompartment();
        }
        return null;
    }

    @Override
    public OMObjectFigure clone() {
        HashMap<Figure, Figure> oldToNew = new HashMap<Figure, Figure>();
        OMObjectFigure that = (OMObjectFigure) super.clone();
        that.setModel((OMObject) this.model.clone());
        return that;
        /*
        that.remap(oldToNew);
        that.nameCompartment = (ListFigure) oldToNew.get(this.nameCompartment);
        that.attributeCompartment = (ListFigure) oldToNew.get(this.attributeCompartment);
        that.setModel((SimulatedObject) this.model.clone());
        that.nameCompartment.getChild(0).addFigureListener(new NameAdapter(that));
        that.layout();
        return that;
         */
    }

    /**
     * Delegate capabilities for storing and retrieving attributes to a
     * CompositeFigure if the encapsulated presentation figure. If no
     * presentation figure is found then the superclass' addAttribute()
     * will be invoked (which currently does not set an attribute).
     * 
     * 
     * @param name	name of the attribute
     * @param value	value associated with this attribute
     */
    @Override
    public void set(AttributeKey name, Object value) {
        if (name.equals(FONT_BOLD)
                || name.equals(FONT_ITALIC)
                || name.equals(FONT_UNDERLINE)) {
        } else {
            super.set(name, value);
            /*
            if (getPresentationFigure() != null) {
            getPresentationFigure().addAttribute(name, value);
            }
            for (Iterator i = getChildren().iterator(); i.hasNext(); ) {
            Figure child = (Figure) i.next();
            child.addAttribute(name, value);
            }
            attributes.put(name, value);
             */
        }
    }

    @Override
    public void attributeAdded(OMClassEvent event) {
        updateAttributeCompartment();
        layout();
    }

    @Override
    public void attributeChanged(OMClassEvent event) {
        updateAttributeCompartment();
        layout();
    }

    @Override
    public void attributeRemoved(OMClassEvent event) {
        updateAttributeCompartment();
        layout();
    }

    @Override
    public void operationAdded(OMClassEvent event) {
    }

    @Override
    public void operationChanged(OMClassEvent event) {
    }

    @Override
    public void operationRemoved(OMClassEvent event) {
    }

    @Override
    public void nameChanged(OMClassEvent event) {
        willChange();
        updateNameCompartment();
        updateAttributeCompartment();
        layout();
        changed();
    }
    @Override
    public void abstractChanged(OMClassEvent event) {

    }

    @Override
    public void classChanged(OMObjectEvent event) {
        synchronized (getLock()) {
            willChange();
            if (clazz != null) {
                clazz.removeSimulatedClassListener(this);
            }
            clazz = model.getSimulatedClass();
            if (clazz != null) {
                clazz.addSimulatedClassListener(this);
            }
            updateNameCompartment();
            updateAttributeCompartment();
            layout();
            changed();
        }
    }

    @Override
    public void nameChanged(OMObjectEvent event) {
        synchronized (getLock()) {
            willChange();
            updateNameCompartment();
            updateAttributeCompartment();
            layout();
            changed();
        }
    }

    @Override
    public void stateChanged(OMObjectEvent event) {
        synchronized (getLock()) {
            willChange();
            updateAttributeCompartment();
            layout();
            changed();
        }
    }

    protected Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    protected ObjectModel getSimulation() {
        return (getDiagram() == null) ? null : (ObjectModel) getDiagram().getSimulation();
    }

    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && model != null) {
            getSimulation().add(getModel());
        }
    }

    @Override
    public void removeNotify(Drawing drawing) {
        if ((drawing instanceof Diagram) && model != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    public int getConnectionCount() {
        return getModel().getRelationships().size();
    }

    public int getConnectionIndex(DiagramFigure f) {
        return getModel().getRelationships().indexOf(f.getModel());
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        updateNameCompartment();
    }

    @Override
    public void generalizationChanged(OMClassEvent event) {
        updateAttributeCompartment();
        layout();
    }

    protected void drawConnectors(Graphics2D g) {
        Rectangle2D.Double r = getBounds();
        Geom.grow(r, 1d, 1d);
        g.setColor(ClassDiagram.OBJECT_CONNECTOR_COLOR);
        g.setStroke(new BasicStroke());
        g.draw(r);
    }

    @Override
    public void read(DOMInput in) throws IOException {
        double x = in.getAttribute("x", 0d);
        double y = in.getAttribute("y", 0d);
        double w = in.getAttribute("w", 0d);
        double h = in.getAttribute("h", 0d);
        setBounds(new Point2D.Double(x, y), new Point2D.Double(x + w, y + h));
        readAttributes(in);
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((OMObject) in.readObject(0));
        in.closeElement();
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        Rectangle2D.Double r = getBounds();
        out.addAttribute("x", r.x);
        out.addAttribute("y", r.y);
        //out.addAttribute("w", r.width);
        //out.addAttribute("h", r.height);
        writeAttributes(out);
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
    }

    @Override
    public void constructorEntered(OMObjectEvent evt) {
    }

    @Override
    public void constructorExited(OMObjectEvent evt) {
    }

    @Override
    public void linkAdded(OMObjectEvent evt) {
    }

    @Override
    public void linkRemoved(OMObjectEvent evt) {
    }

    @Override
    public void methodEntered(OMObjectEvent evt) {
    }

    @Override
    public void methodExited(OMObjectEvent evt) {
    }

    @Override
    public void transmittingMessage(OMObjectEvent evt) {
    }

    @Override
    public void messageTransmitted(OMObjectEvent evt) {
    }

    @Override
    public void transmittingResponse(OMObjectEvent evt) {
    }

    @Override
    public void responseTransmitted(OMObjectEvent evt) {
    }

    @Override
    public int getLayer() {
        return ClassDiagram.OBJECT_LAYER;
    }
}
